#include <QtGui>
#ifdef NEEDGLEE
  #include "GLee.h"
#endif
#include "openglcanvas.h"
#include <cmath>
#include <sys/types.h>  // for fstat, getpwuid, getuid
#include <sys/stat.h>   // for fstat
#ifdef _WIN32
  #include <stdlib.h>   // for getenv (TODO: use getenv_s)
#else
  #include <pwd.h>      // for getpwuid
#endif
#include <cstdio>       // for fopen, fclose, getc
#include <cstring>
#ifdef _MSC_VER
  #include <direct.h>
  #define GET_WORKDIR _getcwd
  #define OPEN_FILE _open
  #define LSEEK_FD _lseek
  #define FOPEN_RO(_descriptor,_filename) \
    if(fopen_s(&(_descriptor),_filename,"r")){ \
        fprintf(stderr,"unable to open file '%s'\n",_filename); \
        exit(-1); \
    }
#else
  // unistd was included above
  #define GET_WORKDIR getcwd
  #define OPEN_FILE open
  #define LSEEK_FD lseek
  #define FOPEN_RO(_descriptor,_filename) \
    _descriptor=fopen(_filename,"r"); \
    if((_descriptor)==NULL){ \
        fprintf(stderr,"unable to open file '%s'\n",_filename); \
        exit(-1); \
    }
#endif

#define PROGNAME "pano_interface"
#define VERT_SHADER_FILE "test_vertex_shader.vert"
#define FRAG_SHADER_FILE "fragment_shader.frag"

#define CONST_PI        3.141592653589793115997963468544185161590576171875
#define CONST_PI_2      1.5707963267948965579989817342720925807952880859375
#define CONST_PI_F      3.1415927410125732421875f
#define CONST_PI_2_F    1.57079637050628662109375f
//#define CONST_PI        (0x1.921fb54442d18p+1)
//#define CONST_PI_2      (0x1.921fb54442d18p+0)
//#define CONST_PI_F      (0x1.921fb6p+1f)
//#define CONST_PI_2_F    (0x1.921fb6p+0f)

OpenGLCanvas::OpenGLCanvas(QWidget *parent) :
    QGLWidget(parent)
{
    setFormat(QGL::DoubleBuffer | QGL::DepthBuffer);
    fov = 60.f;
    fov_max = 60.f; // TODO: I only use floats here because Leo did... check
    scale = 1.0f;
    center_lambda = 0.f;
    center_phi = 0.f;
    fov_scale_relation = "Square Root";
    visualization = "Perspective";
    auto_fov_max=false;

    time_frames = 0;
    time_timer.setInterval(0);
    connect(&time_timer, SIGNAL(timeout()), this, SLOT(slotTimer()));
    time_start = time_time.time();
}

void OpenGLCanvas::slotTimer(void) {
    updateGL();
}

void OpenGLCanvas::change_fov(double f){

    if (fov!=f && f>=1.f && f<=360.f)
        fov=f;
    if (fov<=fov_max)
        scale=1.f;
    //else if (fov>295.f)
    //    scale = 0.02f; // TODO: check this value wrt fov_max
    else {
        if (fov_scale_relation == "Naive")
            scale=fov_max/fov;
        else if (fov_scale_relation == "Square Root")
            scale=sqrt((360.f-fov_max-fov)/(360.-2*fov_max));
        else if (fov_scale_relation == "Linear")
            scale=(360.f-fov_max-fov)/(360.-2*fov_max);
        else if (fov_scale_relation == "Square Power")
            scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),2);
        else if (fov_scale_relation == "Cubic Power")
            scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),3);
        else if (fov_scale_relation == "Logarithm")
            scale=log(exp(1.f)+(1.f-exp(1.f))*(fov-fov_max)/(360.-2*fov_max));
    }

//    scale = 0.3f;

    fprintf(stderr,"change fov, fov=%f, fov_max=%f, scale=%f\n",fov,fov_max,scale);
    emit fov_changed((int)fov);

    updateGL();

}

void OpenGLCanvas::change_fov(int new_fov){
    if(new_fov<=360&&new_fov>=1)
            change_fov((double)new_fov);
    if(auto_fov_max){
        if(new_fov<60)
            change_fov_max(60);
        else
            if(new_fov>180)
                change_fov_max(1);
            else
                change_fov_max(90-new_fov/2);
    }
}

void OpenGLCanvas::change_fov_max(int new_fov_max){
    if(new_fov_max<=360&&new_fov_max>=1)
            fov_max=(double)new_fov_max;
    if (fov<=fov_max)
        scale=1.f;
    //else if (fov>295.f)
    //    scale = 0.02f; // TODO: check this value wrt fov_max
    else {
        if (fov_scale_relation == "Naive")
            scale=fov_max/fov;
        else if (fov_scale_relation == "Square Root")
            scale=sqrt((360.f-fov_max-fov)/(360.-2*fov_max));
        else if (fov_scale_relation == "Linear")
            scale=(360.f-fov_max-fov)/(360.-2*fov_max);
        else if (fov_scale_relation == "Square Power")
            scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),2);
        else if (fov_scale_relation == "Cubic Power")
            scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),3);
        else if (fov_scale_relation == "Logarithm")
            scale=log(exp(1.f)+(1.f-exp(1.f))*(fov-fov_max)/(360.-2*fov_max));
    }
    fprintf(stderr,"change fov_max, fov=%f, fov_max=%f, new scale=%f\n",fov,fov_max,scale);
    emit max_fov_changed((int)fov_max);
    updateGL();
}

//void OpenGLCanvas::change_scale(double s){

//    if (scale!=s && s>=0.0 && s<=1.0) scale = s;
//    updateGL();

//}

void OpenGLCanvas::change_center_lambda(double lambda){

    if (center_lambda!=lambda && lambda>=-CONST_PI && lambda<=CONST_PI) {
        center_lambda = lambda;
        updateGL();
    }

}

void OpenGLCanvas::change_center_phi(double phi){

    if (center_phi!=phi && phi>=-CONST_PI_2 && phi<=CONST_PI_2) {
        center_phi = phi;
        updateGL();
    }

}

void OpenGLCanvas::re_center(){
    center_phi=.0f;
    center_lambda=.0f;
    updateGL();
}

void OpenGLCanvas::change_fov_scale_relation(QString name){

   fov_scale_relation = name;
   if (fov<fov_max) scale = 1.f;
   //else if (fov>295.f) scale = 0.01f;
   else{
       if (fov_scale_relation == "Naive")
           scale=fov_max/fov;
       else if (fov_scale_relation == "Square Root")
           scale=sqrt((360.f-fov_max-fov)/(360.-2*fov_max));
       else if (fov_scale_relation == "Linear")
           scale=(360.f-fov_max-fov)/(360.-2*fov_max);
       else if (fov_scale_relation == "Square Power")
           scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),2);
       else if (fov_scale_relation == "Cubic Power")
           scale=powf((360.f-fov_max-fov)/(360.-2*fov_max),3);
       else if (fov_scale_relation == "Logarithm")
           scale=log(exp(1.f)+(1.f-exp(1.f))*(fov-fov_max)/(360.-2*fov_max));
   }
   fprintf(stderr,"changed scale relation, scale=%f, fov_max=%f\n",scale,fov_max);
   updateGL();

}

void OpenGLCanvas::change_visualization(QString name){

    visualization = name;
    updateGL();

}

// This function reads the contents of the ~/.panorc file and stores the
// options in private variables.
void OpenGLCanvas::read_config_file(){
#ifdef _WIN32
    char* envvar=(char*)malloc(12*sizeof(char));
    strcpy(envvar,"USERPROFILE\0");
    char *filepath=(char*)malloc(512*sizeof(char));
    strcpy(filepath,getenv(envvar));
    free(envvar);
    strcat(filepath,"\\.panorc");
#else
    struct passwd *pw=getpwuid(getuid());
    char *filepath=pw->pw_dir;
    strcat(filepath,"/.panorc");
#endif
    shader_dir=(char*)malloc(512*sizeof(char));
    shader_dir[0]='\0';
    input_image_file=(char*)malloc(512*sizeof(char));
    input_image_file[0]='\0';
    char *read_line=(char*)malloc(64*sizeof(char));
    struct stat testbuf;
    if(stat(filepath,&testbuf)){
        fprintf(stderr,"%s does not exist\n",filepath);
    }else{
        FILE *rcfile;
        FOPEN_RO(rcfile,filepath);
        char c;
        char *line=(char*)malloc(512*sizeof(char));
        while((c=getc(rcfile))!=EOF){
            while(c=='\n') // discard empty lines
                c=getc(rcfile);
            if(c==EOF)
                break;
            line[0]=c; // first char on the line was already read
            if(!fgets(line+1,511,rcfile)){
                fprintf(stderr,"error reading rcfile\n");
                exit(-1);
            }
            // check for 'shader_dir' option
            if(!strncmp(line,"shader_dir=",11)){
                strcpy(shader_dir,line+11);
                shader_dir[strlen(line)-12]='\0';
                fprintf(stderr,"shader_dir=%s\n",shader_dir);
            }
            // check for 'image_file' option
            if(!strncmp(line,"image_file=",11)){
                strcpy(input_image_file,line+11);
                input_image_file[strlen(line)-12]='\0';
                fprintf(stderr,"input_image_file=%s\n",input_image_file);
            }
            // check for 'max_fov' option
            if(!strncmp(line,"max_fov=",8)){
                strcpy(read_line,line+8);
                read_line[strlen(line)-9]='\0';
                fov_max=atof(read_line);
                fprintf(stderr,"max_fov=%f\n",fov_max);
            }
            // check for 'auto_max_fov' option
            if(!strncmp(line,"auto_max_fov=",13)){
                strcpy(read_line,line+13);
                read_line[strlen(line)-14]='\0';
                auto_fov_max=atof(read_line);
                fprintf(stderr,"auto_max_fov=%d\n",auto_fov_max);
            }
        }
        fclose(rcfile);
    }
#ifdef _WIN32
    free(filepath);
#endif
    emit fov_changed((int)fov);
    emit max_fov_changed((int)fov_max);
}

// The file type is determined from the file extension.
enum FileType OpenGLCanvas::get_file_type(const char *filename){
    // find the last '.' in the file name
    int dotposition=-1;
    for(int i=strlen(filename)-1;i>=0;--i){
        if(filename[i]=='.'){
            dotposition=i;
            break;
        }
    }
    // if '.' was not found, the whole string is considered extension
    char *ext=(char*)filename+dotposition+1; // file extension
    if((ext[0]=='p'||ext[0]=='P')&&
       (ext[1]=='n'||ext[1]=='b'||ext[1]=='g'||ext[1]=='p'||
        ext[1]=='N'||ext[1]=='B'||ext[1]=='G'||ext[1]=='P')&&
       (ext[2]=='m'||ext[2]=='M')){
        return PNM;
    }
    if((ext[0]=='j'||ext[0]=='J')&&
       (ext[1]=='p'||ext[1]=='P')&&
       ((strlen(ext)==3&&(ext[2]=='g'||ext[2]=='G'))||
        (strlen(ext)==4&&(ext[2]=='e'||ext[2]=='E')&&
                         (ext[3]=='g'||ext[3]=='G')))){
        return JPEG;
    }
    return UNKNOWN;
}

void OpenGLCanvas::load_image(const char *new_image){
    int width,height;
    const char * const progname=(char*)(PROGNAME);
    unsigned char *textureBytes=NULL;
    int textureSize;
    switch(get_file_type(new_image)){
        case JPEG:
            fprintf(stderr,"input image has JPEG format\n");
            textureSize=jpgGetTextureSize(new_image);
            textureBytes=(unsigned char*)malloc(textureSize);
            jpgReadTextureBytes(new_image,textureBytes,&width,&height);
            break;
        case PNM:
            fprintf(stderr,"input image has PNM format\n");
            textureSize=pnmGetTextureSize(progname,new_image);
            textureBytes=(unsigned char*)malloc(textureSize);
            pnmReadTextureBytes(progname,new_image,textureBytes,&width,&height);
        break;
        default: // UNKNOWN
            fprintf(stderr,"%s: unknown file format\n",new_image);
            exit(-1);
    }
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    GLuint tex;
    glGenTextures(1,&tex);
    glBindTexture(GL_TEXTURE_2D,tex);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
    glTexImage2D(GL_TEXTURE_2D,0, GL_RGB, width,height,0,GL_RGB,GL_UNSIGNED_BYTE,textureBytes);
}

void OpenGLCanvas::change_input_image(){
    load_image(QFileDialog::getOpenFileName(this,tr("Choose Panorama File")).toStdString().c_str());
    updateGL();
}

void OpenGLCanvas::initializeGL(){
    glShadeModel(GL_SMOOTH);
    glClearColor(1.0f,1.0f,1.0f,0.0f);
    glClearDepth(1.0f);
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);

#ifdef __APPLE__
    const char * const progname = "PROJ_ROOT_DIR";
#else
    // progname is a file name or a path???
    const char * const progname = (char*)(PROGNAME);
#endif

    fprintf(stderr,"progname=%s\n",progname);
    read_config_file();
    // If the input file does not exist or was not specified.
    struct stat testbuf;
    if(stat(input_image_file,&testbuf)||!strcmp(input_image_file,"")){
        load_image(QFileDialog::getOpenFileName(this,tr("Choose Panorama File")).toStdString().c_str());
    }else{
        load_image(input_image_file);
    }
    free(input_image_file);

    // mesh resolution
    int m,n;
    m = n = 100;

    //defining texture coordinates
    int meshNumTexCoord = m*n;
    float *texCoord = (float *)malloc(2*meshNumTexCoord*sizeof(float));
    if (texCoord == NULL){
        printf("problem allocating memory for texture coordinates \n");
    }
    define_texture_coordinates(texCoord, m, n, -CONST_PI_2_F, CONST_PI_2_F, -CONST_PI_F, CONST_PI_F);

    //defining positions of the sphere vertices
    int meshNumVertices = m*n;
    float* positions = (float *)malloc(3*meshNumVertices*sizeof(float));
    if (positions == NULL){
        printf("problem allocating memory for positions \n");
    }
//    vertex_transformation(positions, m, n, center_lambda, center_phi, fov_rads, scale); //passar pelo vertex shader
    load_sphere_mesh(positions, m, n); //colocar essa e funcoes para textura e triangulos no initializeGL

    //defining triagle indices
    unsigned int meshNumFaces = 2*(m-1)*(n-1);
    unsigned int meshNumIndices = 3*meshNumFaces;
    unsigned int * indices = (unsigned int *)malloc(meshNumIndices*sizeof(unsigned int));
    define_triangle_indices(indices, m, n);

    // draw setup
    verticesPositions = positions;
    textureCoordinates = texCoord;
    numberOfIndices = meshNumIndices;
    triangleIndices = indices;

    setShaders();
}

void OpenGLCanvas::define_texture_coordinates(float *texCoord, int m, int n, float min_phi, float max_phi, float min_lambda, float max_lambda){

    float delta_lambda = (max_lambda-min_lambda)/(1.0*(n-1));
    float delta_phi = (max_phi-min_phi)/(1.0*(m-1));

    for (int i = 0; i<m; i++){
        for (int j = 0; j<n; j++){
            texCoord[2*(j+i*n)] = (min_lambda+delta_lambda*j)/(2*CONST_PI_F) + 0.5;
            texCoord[2*(j+i*n)+1] = (min_phi+delta_phi*i)/(CONST_PI_F) + 0.5;
        }
    }

}

void OpenGLCanvas::vertex_transformation(float *positions, int m, int n, float center_lambda, float center_phi, float fov_rads, float scale){

    float min_lambda = -CONST_PI_F;
    float max_lambda = CONST_PI_F;
    float min_phi = -CONST_PI_2_F;
    float max_phi = CONST_PI_2_F;

    float delta_lambda = (max_lambda-min_lambda)/(1.0*(n-1));
    float delta_phi = (max_phi-min_phi)/(1.0*(m-1));

    float lambda, phi, x, y, z, u, v, r, theta;

    //calculating the extent of the projection for the given FOV
    lambda = fov_rads;
    phi = 0.f;
    // OpenGL: x is the vertical axes pointg downwards, and y is horizontal axes
    y = sin(phi);
    x = -sin(lambda)*cos(phi);
    z = -cos(lambda)*cos(phi);
    u = 2.f*x/(-z+1.f);
    v = 2.f*y/(-z+1.f);
    r = sqrt(u*u+v*v);
    theta = atan2(u,v);
    r *= scale;
    u = -r*sin(theta);
    v = r*cos(theta);
    x = (4.f*u)/(u*u+v*v+4.f);
    y = (4.f*v)/(u*u+v*v+4.f);
    z = (u*u+v*v-4.f)/(u*u+v*v+4.f);
    u = x/(-z);
    v = y/(-z);
    float extent = u;

    for (int i = 0; i<m; i++){
        for (int j = 0; j<n; j++){

            lambda = (min_lambda+delta_lambda*j);
            phi = (min_phi+delta_phi*i);

            // OpenGL: x is the vertical axes pointg downwards, and y is horizontal axes
            y = sin(phi);
            x = -sin(lambda)*cos(phi);
            z = -cos(lambda)*cos(phi);

            //Rotation 1: (-center_lambda)-rotation on the xz-plane
            float x_copy = x;
            x = cos(-center_lambda)*x - sin(-center_lambda)*z;
            y = 1.f*y;
            z = sin(-center_lambda)*x_copy + cos(-center_lambda)*z;

            //Rotation 2: (-center_phi)-rotation on the yz-plane
            float y_copy = y;
            x = 1.f*x;
            y = cos(-center_phi)*y - sin(-center_phi)*z;
            z = sin(-center_phi)*y_copy + cos(-center_phi)*z;


            u = 2.f*x/(-z+1.f);
            v = 2.f*y/(-z+1.f);

            r = sqrt(u*u+v*v);
            theta = atan2(u,v);

            // scaling the complex plane according to scale specified in the interface (relate it to FOV)
            r *= scale;

            u = -r*sin(theta);
            v = r*cos(theta);

            x = (4.f*u)/(u*u+v*v+4.f);
            y = (4.f*v)/(u*u+v*v+4.f);
            z = (u*u+v*v-4.f)/(u*u+v*v+4.f);

            lambda = atan2(x,-z)/CONST_PI_F;
            phi = asin(y)/CONST_PI_2_F;

            u = x/(-z);
            v = y/(-z);

            if (visualization=="Perspective"){
                positions[3*(j+i*n)] = u/extent;
                positions[3*(j+i*n)+1] = v/extent;
                positions[3*(j+i*n)+2] = z;
            }

            if (visualization=="3D Sphere"){
                positions[3*(j+i*n)] = 0.9f*x;
                positions[3*(j+i*n)+1] = 0.9f*y;
                positions[3*(j+i*n)+2] = z;
            }

            if (visualization=="Equi-Rectangular"){
                positions[3*(j+i*n)] = lambda;
                positions[3*(j+i*n)+1] = phi;
                positions[3*(j+i*n)+2] = z;
            }

        }
    }

}

void OpenGLCanvas::load_sphere_mesh(float *positions, int m, int n){

    float min_lambda = -CONST_PI_F;
    float max_lambda = CONST_PI_F;
    float min_phi = -CONST_PI_2_F;
    float max_phi = CONST_PI_2_F;

    float delta_lambda = (max_lambda-min_lambda)/(1.0*(n-1));
    float delta_phi = (max_phi-min_phi)/(1.0*(m-1));

    float lambda, phi, x, y, z;

    for (int i = 0; i<m; i++){
        for (int j = 0; j<n; j++){

            lambda = (min_lambda+delta_lambda*j);
            phi = (min_phi+delta_phi*i);

            // OpenGL: x is the vertical axes pointg downwards, and y is horizontal axes
            y = sin(phi);
            x = -sin(lambda)*cos(phi);
            z = -cos(lambda)*cos(phi);

            positions[3*(j+i*n)] = x;
            positions[3*(j+i*n)+1] = y;
            positions[3*(j+i*n)+2] = z;

        }
    }

}

float OpenGLCanvas::calculate_extent(float fov_rads){

    double lambda, phi, x, y, z, u, v, r, theta;
    //calculating the extent of the projection for the given FOV
    lambda = fov_rads;
    phi = 0.;
    // OpenGL: x is the vertical axes pointg downwards, and y is horizontal axes
    y = sin(phi);
    x = -sin(lambda)*cos(phi);
    z = -cos(lambda)*cos(phi);
    u = 2.*x/(-z+1.);
    v = 2.*y/(-z+1.);
    r = sqrt(u*u+v*v);
    theta = atan2(u,v);
    r *= scale;
    u = -r*sin(theta);
    v = r*cos(theta);
    x = (4.*u)/(u*u+v*v+4.);
    y = (4.*v)/(u*u+v*v+4.);
    z = (u*u+v*v-4.)/(u*u+v*v+4.);
    u = x/(-z);
    v = y/(-z);
    return u;
}

void OpenGLCanvas::define_triangle_indices(unsigned int * indices, int m, int n){

    for (int i = 0; i<m-1; i++){
        for (int j = 0; j<n-1; j++){

            unsigned int index = (j+i*n);

            indices[3*(2*(j+i*(n-1)))] = index;
            indices[3*(2*(j+i*(n-1)))+1] = index+1;
            indices[3*(2*(j+i*(n-1)))+2] = index+n;

            indices[3*(2*(j+i*(n-1))+1)] = index+1;
            indices[3*(2*(j+i*(n-1))+1)+1] = index+n+1;
            indices[3*(2*(j+i*(n-1))+1)+2] = index+n;

        }
    }

}

int OpenGLCanvas::pnmGetTextureSize(const char * const progname, const char * texturePath)
{
    struct pam inpam;
    pm_init(progname, 0);
    FILE *in_file;
    FOPEN_RO(in_file,texturePath)
#ifdef PAM_STRUCT_SIZE
    pnm_readpaminit(in_file,&inpam,PAM_STRUCT_SIZE(tuple_type));
#else
    pnm_readpaminit(in_file,&inpam,sizeof(struct pam));
#endif
    image_size_x=inpam.width;
    image_size_y=inpam.height;
    int size = image_size_x*image_size_y*inpam.depth*inpam.bytes_per_sample;
    pm_close(in_file);
    return size;
}

void OpenGLCanvas::pnmReadTextureBytes(const char * const progname,
                                          const char * texturePath,
                                          unsigned char * textureBytes,
                                          int * outImageWidth,
                                          int * outImageHeight)
{
        struct pam inpam;
        tuple * tuplerow;
        int row;

        pm_init(progname, 0);
        FILE *in_file;
        FOPEN_RO(in_file,texturePath);

#ifdef PAM_STRUCT_SIZE
        pnm_readpaminit(in_file,&inpam,PAM_STRUCT_SIZE(tuple_type));
#else
        pnm_readpaminit(in_file,&inpam,sizeof(struct pam));
#endif

        tuplerow = pnm_allocpamrow(&inpam);

        for (row = 0; row < inpam.height; row++) {
                int column;
                pnm_readpamrow(&inpam, tuplerow);
                for (column = 0; column < inpam.width; ++column) {
                        unsigned int plane;
                        for (plane = 0; plane < inpam.depth; ++plane) {
                                textureBytes[(inpam.height-row-1)*3*inpam.width+3*column+plane] = tuplerow[column][plane];
                        }
                }
        }

        pnm_freepamrow(tuplerow);

        *outImageWidth = inpam.width;
        *outImageHeight = inpam.height;

        pm_close(in_file);
}

int OpenGLCanvas::jpgGetTextureSize(const char *texturePath)
{
    FILE *in_file;
    FOPEN_RO(in_file,texturePath)
    struct jpeg_decompress_struct cinfo;
    struct jpeg_error_mgr jerr;
    cinfo.err=jpeg_std_error(&jerr);
    jpeg_create_decompress(&cinfo);
    jpeg_stdio_src(&cinfo,in_file);
    jpeg_read_header(&cinfo,0);
    jpeg_start_decompress(&cinfo);
    image_size_x=cinfo.output_width;
    image_size_y=cinfo.output_height;
    int size=image_size_x*image_size_y*cinfo.num_components;
    fclose(in_file);
    return size;
}

void OpenGLCanvas::jpgReadTextureBytes(const char *texturePath,
                                       unsigned char *textureBytes,
                                       int *outImageWidth,
                                       int *outImageHeight)
{
        FILE *in_file;
        FOPEN_RO(in_file,texturePath);
        struct jpeg_decompress_struct cinfo;
        struct jpeg_error_mgr jerr;
        cinfo.err=jpeg_std_error(&jerr);
        JSAMPROW row_pointer[1];
        jpeg_create_decompress(&cinfo);
        jpeg_stdio_src(&cinfo,in_file);
        jpeg_read_header(&cinfo,0);
        jpeg_start_decompress(&cinfo);
        int depth=cinfo.num_components;
        unsigned long scanline=image_size_y;
        row_pointer[0]=(unsigned char*)malloc(image_size_x*depth);
        while(cinfo.output_scanline<cinfo.output_height){
                --scanline;
                jpeg_read_scanlines(&cinfo,row_pointer,1);
                for(int i=0;i<image_size_x*depth;++i){
                    textureBytes[scanline*image_size_x*depth+i]=row_pointer[0][i];
                }
        }
        *outImageWidth=cinfo.output_width;
        *outImageHeight=cinfo.output_height;
        fclose(in_file);
        jpeg_finish_decompress(&cinfo);
        jpeg_destroy_decompress(&cinfo);
}

void OpenGLCanvas::resizeGL(int w, int h){
    if(w>h)
        glViewport(0,(h-w)/2,w,w);
    else
        glViewport((w-h)/2,0,h,h);
}

char * OpenGLCanvas::textFileRead(char *fn) {

    FILE *fp;
    char *content = NULL;
    int f, count;
    f = OPEN_FILE(fn, O_RDONLY);
    count = LSEEK_FD(f, 0, SEEK_END);
//    close(f);
    if (fn != NULL) {
        //fp = fopen(fn,"rt");
        FOPEN_RO(fp,fn);
        if (fp != NULL) {
            if (count > 0) {
                content = (char *)malloc(sizeof(char) * (count+1));
                count = fread(content,sizeof(char),count,fp);
                content[count] = '\0';
            }
            fclose(fp); // maybe this line must be outside the {}
        }
    }
    return content;
}

void OpenGLCanvas::setShaders() {

    char *vs,*fs;

#ifndef __APPLE__
  #ifdef GLEW_VERSION_1_5
    GLenum err=glewInit();
    if(err!=GLEW_OK){
        fprintf(stderr,"error in GLEW initialization: %s\n",glewGetString(err));
        exit(-1);
    }
  #endif
#endif

    GLuint v = glCreateShader(GL_VERTEX_SHADER);
    GLuint f = glCreateShader(GL_FRAGMENT_SHADER);

    // Configure vertex and fragment shader files.
    char *vs_file=(char*)malloc(512*sizeof(char*));
    char *fs_file=(char*)malloc(512*sizeof(char*));
    if(!strcmp(shader_dir,"")){ // if shader_dir was not configured
        if(!GET_WORKDIR(vs_file,512)||!GET_WORKDIR(fs_file,512)){
            fprintf(stderr,"error reading shader files\n");
            exit(-1);
        }
        strcat(vs_file,"/shaders/");
        strcat(fs_file,"/shaders/");
    }else{;
        strcpy(vs_file,shader_dir);
        strcat(vs_file,"/");
        strcpy(fs_file,shader_dir);
        strcat(fs_file,"/");
    }
    strcat(vs_file,VERT_SHADER_FILE);
    strcat(fs_file,FRAG_SHADER_FILE);
    fprintf(stderr,"vs_file=%s\nfs_file=%s\n",vs_file,fs_file);

    struct stat vs_testbuf,fs_testbuf;
    if(stat(vs_file,&vs_testbuf)||stat(fs_file,&fs_testbuf)){
        fprintf(stderr,"a shader file does not exist!\n");
        free(vs_file);
        free(fs_file);
        exit(-1);
    }

    vs=textFileRead(vs_file);
    fs=textFileRead(fs_file);

    const char * vv = vs;
    const char * ff = fs;

    glShaderSource(v, 1, &vv,NULL);
    glShaderSource(f, 1, &ff,NULL);

    free(vs);free(fs);
    free(vs_file);free(fs_file);

    glCompileShader(v);
    glCompileShader(f);

    GLuint p = glCreateProgram();

    glAttachShader(p,v);
    glAttachShader(p,f);

    glLinkProgram(p);
    glUseProgram(p);

}

void OpenGLCanvas::mousePressEvent(QMouseEvent *event){
    lastPos=event->pos();
    //fprintf(stderr,"mouse click\n");
}

void OpenGLCanvas::mouseMoveEvent(QMouseEvent *event){
    // scroll with the left button
    if(event->buttons()==Qt::LeftButton){
        // compute the delta and move the image
        center_lambda+=(event->x()-lastPos.x())*CONST_PI_F/image_size_x;
        center_phi+=(event->y()-lastPos.y())*CONST_PI_F/image_size_y;
        lastPos=event->pos();
        updateGL();
    }
}

void OpenGLCanvas::wheelEvent(QWheelEvent *event){
    if(event->orientation()==Qt::Vertical){
        if(event->modifiers()==Qt::ShiftModifier){
            change_fov_max(fov_max+((double)event->delta())/30);
        }else{
            int new_fov=fov+event->delta()/30;
            change_fov((double)new_fov);
            if(auto_fov_max){
                if(new_fov<60)
                    change_fov_max(60);
                else
                    if(new_fov>180)
                        change_fov_max(1);
                    else
                        change_fov_max(90-new_fov/2);
            }
        }
    }
}

void OpenGLCanvas::paintGL(){

    float fov_rads = (fov/360.)*CONST_PI;

//    // changing scale to generate the figures for the paper (remove it after)
//    scale = 0.8;


    // defining transformation parameters (that will be passed to the vertex shader)
    float extent = calculate_extent(fov_rads);
    float vis_mode=.0;
    if (visualization=="Perspective") vis_mode=1.0;
    else if (visualization=="3D Sphere") vis_mode=2.0;
    else if (visualization=="Equi-Rectangular") vis_mode=3.0;
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, 2.0/extent, 0.0, 2.0/scale, 0.0, -2.0/vis_mode);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glOrtho(0.0, 2.0/center_lambda, 0.0, 2.0/center_phi, -1.0, 1.0);

    // drawing the mesh
    glClearColor(1.0, 1.0, 1.0, 1.0);

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    glColor3f(1, 0, 0);

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, verticesPositions);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(2, GL_FLOAT, 0, textureCoordinates);

    glDrawElements(GL_TRIANGLES, numberOfIndices, GL_UNSIGNED_INT, triangleIndices);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);

    time_frames++;
//    if (time_frames > 0) {
        double dt = time_time.elapsed();
//        if (dt > 0.5) {
            time_fps = time_frames/dt;
            time_frames = 0;
            time_time.reset();
            emit fps(QString("%1 fps").arg((int)(time_fps+0.5)));
            //printf("fps = %d ", (int)(time_fps+0.5));
//        }
//    }

}
